
/* -*-C-*-
 ##############################################################################
 #
 # File:        trice/src/display.c    
 # RCS:         "@(#)$Revision: 1.13 $ $Date: 94/03/09 10:40:28 $"
 # Description: routines from E1430 driver library to display module states 
 # Author:      Doug Passey
 # Created:     
 # Language:    C
 # Package:     E1430
 # Status:      "@(#)$State: Exp $"
 #
 # (C) Copyright 1992, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This files contains all of the routines that handle displaying the contents
 # of E1430 module states.
 #
 # Revisions:
 #
 ##############################################################################
*/


#    include <stdio.h>

#include "trice.h"
#include "err1430.h"

#ifndef lint
const char display_fileId[] = "$Header: display.c,v 1.13 94/03/09 10:40:28 chriss Exp $";
#endif

const char unknownStr[] = "UNKNOWN";

/******************************************************************************
 * Copies current state of a module at logical address, la, into the structure 
 * pointed to by statePtr.  Returns -1 if no module with that logical address
 * exists, otherwise returns 0.
 *****************************************************************************/
SHORTSIZ16 i1430_get_module_int_state(SHORTSIZ16 la, 
						aModuleImage DOS_FAR *statePtr)
{
  int i;

  if(e1430_modStates == NULL || e1430_modStatesSize == 0)  return( -1);

  for(i=0; i < e1430_modStatesSize; i++) {
    if(la == e1430_modStates[i].logicalAddr) {
      *statePtr = e1430_modStates[i];
      return (0);
    }
  }
  return (-1);
}

/******************************************************************************
 * Copies state of a module pointed to by statePtr to the module refered to
 * by logical address, la.  Returns -1 if no module with that logical address
 * exists, otherwise returns 0.
 *****************************************************************************/
SHORTSIZ16 i1430_set_module_int_state(SHORTSIZ16 la, 
					aModuleImage DOS_FAR *statePtr)
{
  int i;

  if(e1430_modStates == NULL || e1430_modStatesSize == 0)  return (-1);

  for(i=0; i < e1430_modStatesSize; i++) {
    if(la == e1430_modStates[i].logicalAddr) {
      e1430_modStates[i] = *statePtr;
      return (0);
    }
  }
  return (-1);
}


/******************************************************************************
 * Converts the current state of a module at logical address, la, into the 
 * the expanded user structure, aE1430State, pointed to by statePtr.  
 * Returns negative error number if error, otherwise returns 0.
 *****************************************************************************/
SHORTSIZ16 e1430_get_module_state(SHORTSIZ16 la, aE1430State DOS_FAR *statePtr)
{
  SHORTSIZ16 index;
  SHORTSIZ16 error;

  for(index=0; index < e1430_modStatesSize; index++) {
    if(la == e1430_modStates[index].logicalAddr) break; 
  }

  if(index == e1430_modStatesSize)  {	/* couldn't find module with logical address */
    return (ERR1430_NO_MOD_AT_LA);
  }

  /* Convert internal state to external state */
  statePtr->logicalAddr = e1430_modStates[index].logicalAddr;

  /* ANALOG */
  statePtr->range = i1430_get_range_index(index);
  statePtr->coupling = i1430_get_coupling_index(index);
  statePtr->inputHigh = i1430_get_input_hi_index(index);
  statePtr->inputLow = i1430_get_input_lo_index(index);
  statePtr->inputFilter = i1430_get_alias_filter_index(index);

  /* DATA FORMAT */
  statePtr->blocksize = i1430_get_blocksize_index(index);
  statePtr->dataMode = i1430_get_data_mode_index(index);
  statePtr->dataPort = i1430_get_data_port_index(index);
  statePtr->dataType = i1430_get_data_type_index(index);
  statePtr->dataSize = i1430_get_data_size_index(index);
  statePtr->appendStatus = i1430_get_append_status_index(index);

  /* TIMING */
  statePtr->syncState = i1430_get_sync_state_index(index);
  statePtr->adcClock = i1430_get_adc_clock_index(index);
  statePtr->dspClock = i1430_get_dsp_clock_index(index);
  e1430_get_clock_master_mode(la, &statePtr->masterState);

  /* TRIGGER */
  statePtr->triggerMode = i1430_get_trigger_source_index(index);
  statePtr->triggerSlope = i1430_get_trigger_slope_index(index);
  statePtr->triggerLevelADC = i1430_get_trig_lev_adc_index(index);
  statePtr->triggerLevelMag = i1430_get_trig_lev_mag_index(index);
  statePtr->triggerDelay = i1430_get_trigger_delay_index(index);
  statePtr->triggerBw = i1430_get_trigger_bw_index(index);

  /* MEASUREMENT */
  statePtr->intrPriority0 = i1430_get_intr_priority_index(index, 0);
  statePtr->intrMask0 = i1430_get_intr_mask_index(index, 0);
  statePtr->intrPriority1 = i1430_get_intr_priority_index(index, 1);
  statePtr->intrMask1 = i1430_get_intr_mask_index(index, 1);

  /* DECIMATION */
  statePtr->decimationBw = i1430_get_decimation_bw_index(index);
  statePtr->decimationState = i1430_get_deci_state_index(index);
  statePtr->decimationOutput = i1430_get_deci_output_index(index);
  statePtr->passTag = i1430_get_pass_tag_index(index);
  
  /* ZOOM */
  statePtr->centerFreq = i1430_get_center_freq_index(index);

  /* MEMORY */
  error = e1430_get_fifo_size(la, &(statePtr->memorySize));
  if(error) return (error);

  return (0);
}

/******************************************************************************
 * get the name of a write register image if you know its offset.
 * Returns negative error number if error, otherwise it returns 0.
 *****************************************************************************/
char *i1430_get_register_name(SHORTSIZ16 reg, SHORTSIZ16 card)
{

/************** PROBLEM, what to do about read/write registers???? */

  if(card == E1430_CARD_REGISTER) {	/* read registers from card */
    switch(reg) {
      case E1430_VXI_ID_REG:
	return ("VXI_ID_REG");
      case E1430_VXI_DEVICE_TYPE_REG:
	return ("VXI_DEVICE_TYPE_REG");
      case E1430_VXI_STATUS_REG:
	return ("VXI_CONTROL/STATUS_REG");
      case E1430_HP_COMMON_REG:
	return ("HP_COMMON_REG");
      case E1430_HP_DESCRIPTION_REG:
	return ("HP_DESCRIPTION_REG");
      case E1430_HP_SUBCLASS_REG:
	return ("HP_SUBCLASS_REG");
      case E1430_HP_SEND_DATA_REG:
	return ("HP_SEND_DATA_REG");
      case E1430_DECIMATION_CNTLAT_0_REG:
	return ("DECIMATION_CNTLAT_0_REG");
      case E1430_DECIMATION_CNTLAT_1_REG:
	return ("DECIMATION_CNTLAT_1_REG");
      case E1430_DECIMATION_CNTLAT_2_REG:
	return ("DECIMATION_CNTLAT_2_REG");
      case E1430_DECIMATION_CNTLAT_3_REG:
	return ("DECIMATION_CNTLAT_3_REG");
      case E1430_LO_TRANSFER_REG:
	return ("LO_TRANSFER_REG");
    }
  }else{			/* read image registers */
    switch(reg) {
      case E1430_ANALOG_SETUP_REG:
	return ("ANALOG_SETUP_REG");
      case E1430_INPUT_OFFSET_REG:
	return ("INPUT_OFFSET_REG");
      case E1430_MEAS_CONTROL_REG:
	return ("MEAS_CONTROL_REG");
      case E1430_DATA_FORMAT_REG:
	return ("DATA_FORMAT_REG");
      case E1430_IRQ_CONFIG_0_REG:
	return ("IRQ_CONFIG_0_REG");
      case E1430_IRQ_CONFIG_1_REG:
	return ("IRQ_CONFIG_1_REG");
      case E1430_HP_PORT_CONTROL_REG:
	return ("HP_PORT_CONTROL_REG");
      case E1430_TRIGGER_SETUP_REG:
	return ("TRIGGER_SETUP_REG");
      case E1430_TRIGGER_BLOCKSIZE_REG:
	return ("TRIGGER_BLOCKSIZE_REG");
      case E1430_TRIGGER_OFFSET_LO_REG:
	return ("TRIGGER_OFFSET_LO_REG");
      case E1430_TRIGGER_PASSOUT_REG:
	return ("TRIGGER_PASSOUT_REG");
      case E1430_TIMING_SETUP_REG:
	return ("TIMING_SETUP_REG");
      case E1430_ADC_CONTROL_REG:
	return ("ADC_CONTROL_REG");
      case E1430_ZOOM_PHASE_0_REG:
	return ("ZOOM_PHASE_0_REG");
      case E1430_ZOOM_PHASE_1_REG:
	return ("ZOOM_PHASE_1_REG");
      case E1430_ZOOM_INTERP_0_REG:
	return ("ZOOM_INTERP_0_REG");
      case E1430_ZOOM_INTERP_1_REG:
	return ("ZOOM_INTERP_1_REG");
      case E1430_ZOOM_INTERP_2_REG:
	return ("ZOOM_INTERP_2_REG");
      case E1430_ZOOM_INTERP_3_REG:
	return ("ZOOM_INTERP_3_REG");
      case E1430_ZOOM_CONTROL_REG:
	return ("ZOOM_CONTROL_REG");
      default:
	return("INVALID REGISTER OFFSET");
    }
  }
  return(NULL);
}


/******************************************************************************
 * Displays the external state structure of all modules in the module group,
 * groupID using printfs.  Returns ERR1430_NO_GROUP is groupID is invalid, 
 * otherwise it returns 0.
 *****************************************************************************/
SHORTSIZ16 e1430_display_module_state(SHORTSIZ16 groupID)
{
  aModGroupPtr p;
  const char *enumStr;
  aE1430State state;

  p = i1430_valid_module_group(groupID);
  if(p == NULL)  {	 /* check for bad group id */
    return (ERR1430_NO_GROUP);
  }

  for(; *p != -1; p++) {  /* for every module in group */
    (void) e1430_get_module_state(e1430_modStates[*p].logicalAddr, &state);

    (void)printf("\n********************************************\n\n"); 
    (void)printf("Logical address:      %d\n", state.logicalAddr);

    (void)printf("\n***** ANALOG *****\n");
    (void)printf("Range:                %f\n", state.range);
    
    switch(state.coupling) {
      case E1430_COUPLING_AC:
	enumStr = "AC";
	break;
      case E1430_COUPLING_DC:
	enumStr = "DC";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Coupling:             %s\n", enumStr);
    
    switch(state.inputHigh) {
      case E1430_INPUT_HI_CONN:
	enumStr = "Connector";
	break;
      case E1430_INPUT_HI_GROUND:
	enumStr = "Ground";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Input high source:         %s\n", enumStr);
    
    switch(state.inputLow) {
      case E1430_INPUT_LO_FLOAT:
	enumStr = "Floating";
	break;
      case E1430_INPUT_LO_GROUND:
	enumStr = "Grounded";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Input connector shield:         %s\n", enumStr);
    
    switch(state.inputFilter) {
      case E1430_ANTIALIAS_ON:
	enumStr = "On";
	break;
      case E1430_ANTIALIAS_OFF:
	enumStr = "Off";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Anti-alias filter:    %s\n", enumStr);


    (void)printf("\n***** DATA FORMAT *****\n");

    (void)printf("Block size:           %ld\n", state.blocksize);           

    switch(state.dataMode) {
      case E1430_BLOCK_MODE:
	enumStr = "Block";
	break;
      case E1430_CONTINUOUS_MODE:
	enumStr = "Continuous";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Data mode:            %s\n", enumStr);

    switch(state.dataPort) {
      case E1430_SEND_PORT_VME:
	enumStr = "VME Bus";
	break;
      case E1430_SEND_PORT_LBUS:
	enumStr = "Local Bus";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Data port:            %s\n", enumStr);
    
    switch(state.appendStatus) {
      case E1430_APPEND_STATUS_OFF:
	enumStr = "Off";
	break;
      case E1430_APPEND_STATUS_ON:
	enumStr = "On";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Append Status:        %s\n", enumStr);

    switch(state.dataType) {
      case E1430_DATA_TYPE_REAL:
	enumStr = "Real";
	break;
      case E1430_DATA_TYPE_COMPLEX:
	enumStr = "Complex";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Data type:            %s\n", enumStr);
    
    switch(state.dataSize) {
      case E1430_DATA_SIZE_16:
	enumStr = "16 bits";
	break;
      case E1430_DATA_SIZE_32:
	enumStr = "32 bits";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Data size:            %s\n", enumStr);
    
    
    (void)printf("\n***** TIMING *****\n");

    switch(state.syncState) {
      case E1430_MULTI_SYNC_ON:
	enumStr = "Synchronous";
	break;
      case E1430_MULTI_SYNC_OFF:
	enumStr = "Non-Synchronous";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Sync state:           %s\n", enumStr);
    
    switch(state.adcClock) {
      case E1430_ADC_CLOCK_EXTERNAL:
	enumStr = "External";
	break;
      case E1430_ADC_CLOCK_INTERNAL:
	enumStr = "Internal";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("ADC clock:            %s\n", enumStr);

    switch(state.dspClock) {
      case E1430_DSP_CLOCK_ADC:
	enumStr = "ADC";
	break;
      case E1430_DSP_CLOCK_INTERNAL:
	enumStr = "Internal";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("DSP clock:            %s\n", enumStr);

    switch(state.masterState) {
      case E1430_MASTER_CLOCK_ON:
	enumStr = "ACTIVATED by module";
	break;
      case E1430_MASTER_CLOCK_OFF:
	enumStr = "NOT ACTIVATED by module";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Master clock and SYNC:            %s\n", enumStr);


    (void)printf("\n***** TRIGGER *****\n");

    switch(state.triggerMode) {
      case E1430_TRIGGER_SOURCE_OFF:
	enumStr = "Off";
	break;
      case E1430_TRIGGER_SOURCE_AUTO:
	enumStr = "Auto";
	break;
      case E1430_TRIGGER_SOURCE_EXT:
	enumStr = "External";
	break;
      case E1430_TRIGGER_SOURCE_ADC:
	enumStr = "ADC";
	break;
      case E1430_TRIGGER_SOURCE_MAG:
	enumStr = "Magnitude";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Trigger Mode:         %s\n", enumStr);

    switch(state.triggerSlope) {
      case E1430_TRIGGER_SLOPE_POS:
	enumStr = "Positive";
	break;
      case E1430_TRIGGER_SLOPE_NEG:
	enumStr = "Negative";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Trigger Slope:        %s\n", enumStr);

    (void)printf("ADC Trigger Level:        %f volts\n", 
			state.triggerLevelADC * state.range);

    (void)printf("Mag Trigger Level:        %f dB\n", state.triggerLevelMag);

    (void)printf("Trigger Delay:        %ld\n", state.triggerDelay);

    if(state.triggerBw) {		/* non-zero */
      (void)printf("Trigger Bandwidth:    Fs/%d\n", 2L << state.triggerBw);
    }else{
      (void)printf("Trigger Bandwidth:    Full span\n");
    }

    (void)printf("\n***** MEASUREMENT *****\n");

    (void)printf("Interrupt 0 priority: %d\n", state.intrPriority0);

    (void)printf("Interrupt 0 mask:     0x%0.4hX\n", state.intrMask0);

    (void)printf("Interrupt 1 priority: %d\n", state.intrPriority1);

    (void)printf("Interrupt 1 mask:     0x%0.4hX\n", state.intrMask1);


    (void)printf("\n***** DECIMATION *****\n");
    (void)printf("Decimation bandwidth: ");
    if(state.decimationBw == 0) {	/* filter is off */
      (void)printf("OFF");
    }else if(state.decimationBw > 25) {  /* illegal */
      (void)printf("%s", unknownStr);
    }else {					/* everything looks ok */
      (void)printf("Fs/2^N   N = %d", state.decimationBw);
    }
    (void)printf("\n");

    switch(state.decimationState) {
      case E1430_DECIMATION_ON:
	enumStr = "ON";
	break;
      case E1430_DECIMATION_OFF:
	enumStr = "OFF";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Decimation:           %s\n", enumStr);

    switch(state.decimationOutput) {
      case E1430_ONEPASS:
	enumStr = "Single pass";
	break;
      case E1430_MULTIPASS:
	enumStr = "Multi pass";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Decimation output:    %s\n", enumStr);
    
    switch(state.passTag) {
      case E1430_PASS_TAG_24:
	enumStr = "24";
	break;
      case E1430_PASS_TAG_32:
	enumStr = "32";
	break;
      default:
	enumStr = unknownStr;
	break;
    }
    (void)printf("Pass Tag:             %s\n", enumStr);
    
    
    (void)printf("\n***** ZOOM *****\n");

    (void)printf("Center frequency:     %f\n", state.centerFreq);

    (void)printf("\n***** MEMORY *****\n");

    (void)printf("FIFO memory size:     %ld\n", state.memorySize);

    (void)printf("\n********************************************\n\n");
  } /* for p */
  return (0);
}


/******************************************************************************
 * Displays the contents of a VXI register at offset, <offset>, for all
 * modules in <groupID>.
 * Returns negative error number if error, otherwise it returns 0.
 *****************************************************************************/
SHORTSIZ16 e1430_display_register(SHORTSIZ16 groupID, SHORTSIZ16 offset, 
						SHORTSIZ16 which)
{
  SHORTSIZ16 data;
  SHORTSIZ16 la;
  aModGroupPtr p;
  char *str;

  p = i1430_valid_module_group(groupID);
  if(p == NULL)  {	 /* check for bad group id */
    return (ERR1430_NO_GROUP);
  }

  str = i1430_get_register_name(offset, which);

  for(; *p != -1; p++) {
    la = e1430_modStates[*p].logicalAddr;
    if(which == E1430_CARD_REGISTER) {		/* read register on card */
      (void)e1430_read_register_card(la, offset, &data);
    }else{					/* read image register */
      e1430_read_register_image(la, offset, &data);
    }
    (void)printf("la = %d   %s = %0.4XH\n", la, str, data);
  }

  return (0);
}

/******************************************************************************
 * Displays the internal state structure of all modules in the module group,
 * groupID using printfs.  Returns ERR1430_NO_GROUP is groupID is invalid, 
 * otherwise it returns 0.
 *****************************************************************************/
SHORTSIZ16 e1430_display_module_int_state(SHORTSIZ16 groupID)
{
  aModGroupPtr p;
  aModuleImage *ptr;

  p = i1430_valid_module_group(groupID);
  if(p == NULL)  {	 /* check for bad group id */
    return (ERR1430_NO_GROUP);
  }

  (void)printf("\n*************************************************\n");
  (void)printf("Internal state of module group ID %d:\n\n", groupID);
  for(; *p != -1; p++) {  /* for every module in group */
    ptr = &(e1430_modStates[*p]);
    (void)printf("Logical Address:      %d\n", ptr->logicalAddr);
    (void)printf("Mainframe number:     %d\n", ptr->cage);
    (void)printf("Slot number:          %d\n", ptr->slot);
    (void)printf("Analog Setup:         0x%0.4hX\n", ptr->analogSetup);
    (void)printf("Input offset:         %d\n", ptr->offset);
    (void)printf("Measurement Control:  0x%0.4hX\n", ptr->measControl);
    (void)printf("Data Format:          0x%0.4hX\n", ptr->dataFormat);
    (void)printf("IRQ Config 0:         0x%0.4hX\n", ptr->irqConfig0);
    (void)printf("IRQ Config 1:         0x%0.4hX\n", ptr->irqConfig1);
    (void)printf("Port Control:         0x%0.4hX\n", ptr->portControl);
    (void)printf("Trigger Setup:        0x%0.4hX\n", ptr->trigSetup);
    (void)printf("Trigger Blocksize:    0x%0.4hX\n", ptr->trigBlocksize);
    (void)printf("Trigger Offset:       0x%0.4hX\n", ptr->trigOffset);
    (void)printf("Trigger Passout:      0x%0.4hX\n", ptr->trigPassout);
    (void)printf("Timing Setup:         0x%0.4hX\n", ptr->timingSetup);
    (void)printf("ADC Control:          0x%0.4hX\n", ptr->adcControl);
    (void)printf("Zoom Phase 0:         0x%0.4hX\n", ptr->zoomPhase0);
    (void)printf("Zoom Phase 1:         0x%0.4hX\n", ptr->zoomPhase1);
    (void)printf("Zoom Interp 0:        0x%0.4hX\n", ptr->zoomInterp0);
    (void)printf("Zoom Interp 1:        0x%0.4hX\n", ptr->zoomInterp1);
    (void)printf("Zoom Interp 2:        0x%0.4hX\n", ptr->zoomInterp2);
    (void)printf("Zoom Interp 3:        0x%0.4hX\n", ptr->zoomInterp3);
    (void)printf("Zoom Control:         0x%0.4hX\n", ptr->zoomControl);
    (void)printf("\n");
  }
  return (0);
}
